require 'benchmark'
module ActiveRecord
  module ConnectionAdapters # :nodoc:
    # ActiveRecord supports multiple database systems. AbstractAdapter and
    # related classes form the abstraction layer which makes this possible.
    # An AbstractAdapter represents a connection to a database, and provides an
    # abstract interface for database-specific functionality such as establishing
    # a connection, escaping values, building the right SQL fragments for ':offset'
    # and ':limit' options, etc.
    #
    # All the concrete database adapters follow the interface laid down in this class.
    # ActiveRecord::Base.connection returns an AbstractAdapter object, which
    # you can use.
    #
    # Most of the methods in the adapter are useful during migrations. Most
    # notably, the instance methods provided by SchemaStatement are very useful.
    class AbstractAdapter
     
      protected
        def log(sql, name)
          if block_given?
            result = nil
            ms = Benchmark.ms { result = yield }
            @runtime += ms
            log_info(sql, name, ms)
            result
          else
            log_info(sql, name, 0)
            nil
          end
        rescue Exception => e
          # Log message and raise exception.
          # Set last_verification to 0, so that connection gets verified
          # upon reentering the request loop
          @last_verification = 0
          message = "#{utf8ForRuby191(e.class.name)}: #{utf8ForRuby191(e.message)}: #{utf8ForRuby191(sql)}"
          log_info(message, name, 0)
          raise ActiveRecord::StatementInvalid, message
        end

        def utf8ForRuby191(str)
          str.force_encoding(Encoding::UTF_8) if str.respond_to?(:force_encoding)
        end
    end
  end
end
