#!/usr/bin/ruby
# -*- coding: utf-8 -*-
#
# e-PRTR XML-3 Export
# (c) 2009, 2010, 2011, 2012, 2016 Matthias Lüttgert, ENDA GmbH & Co. KG
# (c) 2009 Matthias Lüttgert, Torsten Lüdtke, RISA GmbH
#
# This program is free software; you can redistribute it and/or modify it under
# the terms of the GNU General Public License as published by the Free Software
# Foundation; either version 3 of the License, or (at your option) any later
# version.
#
# This program is distributed in the hope that it will be useful, but WITHOUT ANY
# WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A
# PARTICULAR PURPOSE. See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License along with
# this program; if not, see <http://www.gnu.org/licenses/>.
#
# DEUTSCH:
#
# Dieses Programm ist freie Software. Sie können es unter den Bedingungen der GNU
# General Public License, wie von der Free Software Foundation veröffentlicht,
# weitergeben und/oder modifizieren, entweder gemäß Version 3 der Lizenz oder
# (nach Ihrer Option) jeder späteren Version.
#
# Die Veröffentlichung dieses Programms erfolgt in der Hoffnung, daß es Ihnen von
# Nutzen sein wird, aber OHNE IRGENDEINE GARANTIE, sogar ohne die implizite
# Garantie der MARKTREIFE oder der VERWENDBARKEIT FÜR EINEN BESTIMMTEN
# ZWECK. Details finden Sie in der GNU General Public License.
#
# Sie sollten ein Exemplar der GNU General Public License zusammen mit diesem
# Programm erhalten haben. Falls nicht, siehe <http://www.gnu.org/licenses/>.
#
#
# Purpose
# Read Data from e-PRTR Database an save them to e-PRTR XML-3 format or don't do
# this or partially don't do this or do it in a different way than expected.
#
#
# Changes
# 2016-05-27 mlt: Migration ActiveRecord v2 auf v4 und Ruby v1.9 auf v2.3
# 2012-04-24 mlt: Möglichkeit geschaffen, zu unterdrückende Betriebe durchzulassen
# ????-??-?? rmi: Umstellung von bvmet auf äh, wie hieß das gleich Rory?
# 2011-03-24 mlt: Implementation of CO2 EXCL BIOMASS Export (nbiofracht)
# 2011-03-23 mlt: Split up main export function into several, handleable functions
# 2011-03-22 mlt: Introduced table style error logs for quality and handling improvement
# 2010-11-02 mlt: Introduced lookup for predecessor plants and their ID for correct
#                 export of PreviousNationalID

begin

  require 'rubygems'
  require 'xml/libxml'
  # require 'iconv'
  require 'logger'
  require 'csv'


  load 'db_controller.rb'
  load 'db_classes.rb'
  load 'gk2geo.rb'
  load '../../../ruby/free_ruby_enda/lib/stnd.rb'

  APPNAME = "E-PRTR-Database-Export to XML3"
  APPVERSION = "1.2.0"
  ABOUT = <<EOS



(c) ENDA, 2016, 2012, 2011, 2010, 2009; RISA GmbH 2009

Besuchen Sie die Homepage der Firma ENDA unter
https://enda.eu
für Informationen über die ENDA GmbH & Co. KG
EOS

  IO.popen("id -nu") {|f| $user = f.gets.chomp; puts $user}
  $SHARES_PATH = "/home/#{$user}"

  if $user == 'mlt'
    $PROJECT_PATH = "#{$SHARES_PATH}/dev/ptah/prj/e_prtr_kampa_t2016_iteration2"
  else
    $PROJECT_PATH = "#{$SHARES_PATH}/prj/e_prtr_kampa_t2016_iteration2"
  end


  class String
    def to_u
      # conv = Iconv.new("UTF-8", "ISO-8859-1")
      # return conv.iconv(self)
      return self.encode("UTF-8", "ISO-8859-1")
    end

    def to_iso
      # conv = Iconv.new("ISO-8859-1", "UTF-8")
      # return conv.iconv(self)
      return self.encode("ISO-8859-1", "UTF-8")
    end
  end

  class Float
    def to_3signdigits

      f = ("%.3G" % self).to_f

      if f >= 100
        s = "%.0f" % f
      elsif f >= 10
        s = "%.1f" % f
      elsif f >= 1
        s = "%.2f" % f
      elsif f >= 0.1
        s = "%.3f" % f
      elsif f >= 0.01
        s = "%.4f" % f
      elsif f >= 0.001
        s = "%.5f" % f
      elsif f >= 0.0001
        s = "%.6f" % f
      elsif f >= 0.00001
        s = "%.7f" % f
      elsif f >= 0.000001
        s = "%.8f" % f
      elsif f >= 0.0000001
        s = "%.9f" % f
      else
        # s = "%.10f" % f
        s = "%.2f" % f
      end

      return s
    end
  end

  module Xml_utilities
    def simple_node_creator (node, node_name)
      node << new_node = XML::Node.new(node_name)
      return new_node
    end

    def simple_node_with_content_creator (node, node_name, content)
      node << new_node = XML::Node.new(node_name)
      new_node << content
    end

    def simple_node_with_attribut_creator (node, node_name, attribut,
        attributvalue)

      node << new_node = XML::Node.new(node_name)
      new_node[attribut] = attributvalue
      return new_node
    end

    def simple_node_with_attribut_and_content_creator (node, node_name, attribut,
        attributvalue, content)

      node << new_node = XML::Node.new(node_name)
      new_node[attribut] = attributvalue
      new_node << content
    end

    def translate_de2eu(eucodename, schl, node, node_name, kennnr, log)
      translation = Translationlist.where(
        "state = 0 and eucodename = ? and schl like ? and " +
        "(date_from <= ? or date_from is null) and " +
        "(date_to >= ? or date_to is null)",
        eucodename, schl, @zeitstempel, @zeitstempel).first
      reval = nil
      unless translation == nil
        if translation.eucode?
          simple_node_with_content_creator(node,
            node_name, translation.eucode.to_s)
          reval = translation.eucode.to_s
        else
          log.error("Fehler in Knoten '" +
              node_name + "' für Kennnummer '" +
              kennnr + "'! Für den deutschen Schlüssel '" +
              schl + "' gibt es keinen EU-Schlüssel.")
        end
      else
        log.error("Fehler in Knoten '" +
            node_name + "' für Kennnummer '" +
            kennnr + "'! Schlüssel '" +
            schl + "' konnte nicht in EU-Schlüssel '" +
            eucodename + "' umgewandelt werden.")
      end
      reval
    end

    def translate_de2eu2(eucodename, schl, node, node_name1, node_name2, kennnr, log)
      translation = Translationlist.where(
        "state = 0 and eucodename = ? and schl like ? and " +
        "(date_from <= ? or date_from is null) and " +
        "(date_to >= ? or date_to is null)",
        eucodename, schl, @zeitstempel, @zeitstempel).first
                                                      
      unless translation == nil
        if translation.eucode?
          simple_node_with_content_creator(node,
            node_name1, translation.eucode.to_s)
          simple_node_with_content_creator(node,
            node_name2, translation.eutext.to_s) if translation.eutext?
        else
          log.error("Fehler in Knoten '" +
              node_name1 + "' für Kennnummer '" +
              kennnr + "'! Für den deutschen Schlüssel '" +
              schl + "' gibt es keinen EU-Schlüssel.")
        end
      else
        log.error("Fehler in Knoten '" +
            node_name1 + "' für Kennnummer '" +
            kennnr + "'! Schlüssel '" +
            schl + "' konnte nicht in EU-Schlüssel '" +
            eucodename + "' umgewandelt werden.")
      end
    end
  end

  class Exporter
    include Xml_utilities

    def initialize (path, xsdfile, year, name, street, streetnumber, postcode,
        city, telephone, fax, email, contact, land_id)
      @path         = path
      @xsdfile      = xsdfile
      @year         = year
      @name         = name
      @street       = street
      @streetnumber = streetnumber
      @postcode     = postcode
      @city         = city
      @telephone    = telephone
      @fax          = fax
      @email        = email
      @contact      = contact
      @zeitstempel  = year.to_s + "0101"
      @land_id      = land_id
    end

    def prepare
      bl = Bundesland.find(@land_id)
      blname = bl.ktext
      @logfile = @path + '/xml3-' + blname.to_iso + '.log'
      @log_designation_file = @path + '/xml3-designation' + blname.to_iso + '.log'
      @log_table_file = @path + '/xml3_errortable.csv'
      @xmlfile = @path + '/xml3-' + blname.to_iso + '.xml'

      File.delete(@xmlfile) if File.exists?(@xmlfile)
      File.delete(@log_designation_file) if File.exists?(@log_designation_file)
      File.delete(@log_table_file) if File.exists?(@log_table_file)
      File.delete(@logfile) if File.exists?(@logfile)

      @odoc = XML::Document.new('1.0')
      # @odoc.encoding = 'UTF-8'
      @odoc.encoding = XML::Encoding::UTF_8

      @odoc.root = XML::Node.new('rsm:PollutantReleaseAndTransferReport')
      @oroot = @odoc.root
      @oroot['xmlns:rsm'] = "urn:eu:com:env:prtr:data:standard:2".to_u
      @oroot['xmlns:xsi'] = "http://www.w3.org/2001/XMLSchema-instance".to_u
      @oroot['xsi:schemaLocation'] = "urn:eu:com:env:prtr:data:standard:2 " +
        "http://www.eionet.europa.eu/schemas/eprtr/PollutantReleaseAndTransferReport_2p0.xsd".to_u

      simple_node_with_content_creator(@oroot, "rsm:ReportingYear", @year)
      simple_node_with_content_creator(@oroot, "rsm:CountryID", "DE")
      simple_node_with_content_creator(@oroot, "rsm:CoordinateSystemID", "EPSG:4326")
      simple_node_with_content_creator(@oroot, "rsm:RemarkText",
        "Export from German E-PRTR Database with Ruby")

      competentauthorityparty = simple_node_creator(@oroot,
        "rsm:CompetentAuthorityParty")
      simple_node_with_content_creator(competentauthorityparty, "rsm:Name", @name)

      adress = simple_node_creator(competentauthorityparty, "rsm:Address")
      simple_node_with_content_creator(adress, "rsm:StreetName", @street)
      simple_node_with_content_creator(adress, "rsm:BuildingNumber", @streetnumber)
      simple_node_with_content_creator(adress, "rsm:CityName", @city)
      simple_node_with_content_creator(adress, "rsm:PostcodeCode", @postcode)

      telephone = simple_node_creator(competentauthorityparty, "rsm:TelephoneCommunication")
      simple_node_with_content_creator(telephone, "rsm:CompleteNumberText", @telephone)

      fax = simple_node_creator(competentauthorityparty, "rsm:FaxCommunication")
      simple_node_with_content_creator(fax, "rsm:CompleteNumberText", @fax)

      email = simple_node_creator(competentauthorityparty, "rsm:EmailCommunication")
      simple_node_with_content_creator(email, "rsm:EmailURIID", @email)

      simple_node_with_content_creator(competentauthorityparty, "rsm:ContactPersonName", @contact)
      return true
    end

    def validate
      schemafile = open(@xsdfile) { |f| f.read }
      schema = XML::Schema.from_string(schemafile)

      if !schema
        puts "Schema konnte nicht geladen werden."
        return false
      end

      puts "Zeit: "+ now + " vor Validierung der Eingabe"

      if !@xmldoc.validate_schema(schema)
        puts "ACHTUNG! Die Datei <#{@xmlfile}> entspricht nicht dem Schema <#{@xsdfile}>!"
        puts "Die Konvertierung kann nicht vorgenommen werden."
        return false
      else
        puts "Erfolg: Die Datei <#{@xmlfile}> entspricht dem Schema <#{@xsdfile}>!"
        return true
      end
    end
    

    def export_betrieb_einzeln( b, betriebsname )
      facilityreport = simple_node_creator(@oroot, "rsm:FacilityReport")
      
      simple_node_with_content_creator(facilityreport, "rsm:NationalID", b.kennnr)
      
      previousnationalid = simple_node_creator(facilityreport, "rsm:PreviousNationalID")
      # betriebsvorgaenger = BetriebVorgaenger.find(:all, :conditions => ["state = 0 and p_betrieb_id = ?", b.id])
      betriebsvorgaenger = BetriebVorgaenger.where("state = 0 and p_betrieb_id = ?", b.id)
      vorgaenger = ''
      vorjahr = @year - 1
      betriebsvorgaenger.each do |bvg|
        if vorbetrieb = Betrieb.where("id = ?", bvg.p_betrieb_vorgaenger_id).first
          # Leider kann die EU immer nur EINEN Vorgänger verarbeiten. Wurde der Betrieb geteilt, darf nur einer der Nachfolger
          # auf den Vorgänger verweisen! Also muss geprüft werden, ob die Kennnummer des Vorgängers als aktive Kennnummer in
          # Benutzung in einem anderen Betrieb (als dem gerade verarbeiteten) ist. Dann darf dieser Nachfolger auf keinen Fall
          # auf seinen Vorgänger verweisen.
          vorbetrieb_heute = Betrieb.where("state = 0 and xml3export = 1 and process is null and jahr = '?' and kennnr = ?", @year, vorbetrieb.kennnr).first
          if !vorbetrieb_heute
            # Aber auch, wenn die Vorgänger-Kennnummer nicht mehr in Gebrauch ist, könnten mehrere Nachfolger auf denselben
            # Vorgänger zeigen. Auch das gilt es zu verhindern!
            andere_nachfolger_des_vorgaengers = BetriebVorgaenger.where("state = 0 and p_betrieb_vorgaenger_id = ? and p_betrieb_id <> ?", vorbetrieb.id, b.id)
            if andere_nachfolger_des_vorgaengers.length > 0
              puts "    EU-Unterdrückung wegen Unvermögens, mehrere Vorgänger zu verarbeiten, aktiv! Vorbetrieb: #{vorbetrieb.kennnr}"
              andere_nachfolger_des_vorgaengers.each do |anf|
                nachbetrieb = Betrieb.where("id = ?", anf.p_betrieb_id).first
                puts "Nachfolger: #{nachbetrieb.kennnr} (Nachfolger-ID: #{anf.p_betrieb_id})"
              end
              next
            end
          end
          if vorbetrieb_heute || andere_nachfolger_des_vorgaengers.length > 0
            puts "    EU-Unterdrückung wegen Unvermögens, mehrere Vorgänger zu verarbeiten, aktiv! Vorbetrieb: #{vorbetrieb.kennnr}"
            next
          end
          vorgaenger += vorbetrieb.kennnr + ", "
          vorjahr = vorbetrieb.jahr
        end
      end
      vorgaenger.chomp!(', ')
      if vorgaenger.length > 0 && vorgaenger != b.kennnr
        simple_node_with_content_creator(previousnationalid, "rsm:NationalID", vorgaenger)
        puts "    PreviousNationalID/NationalID differs: <#{vorgaenger}>"
      else
        simple_node_with_content_creator(previousnationalid, "rsm:NationalID", b.kennnr)
        if vorgaenger.length > 0
          puts "    ACHTUNG: Vorgänger mit identischer Kennnummer existiert"
        end
      end
      # TODO: Hier muss der Bericht aus den letzten Reportjahren rausgesucht werden und das entsprechende Jahr eingetragen werden
      # Es kann aber sein, dass der Betrieb überhaupt noch nicht berichtet wurde, dann wissen wir noch nicht, wie sich UBA und
      # COM geeinigt haben.
      simple_node_with_content_creator(previousnationalid,
                                       "rsm:ReportingYear", vorjahr.to_s)
      
      bgk = BetriebGK.where("state = 0 and p_betrieb_id = ?", b.id).first
      
      if b.geheimpd == 1 and b.geheimge == 1 and b.geheimbg == 1
        freigabe = 1
      else
        freigabe = 0
      end
      
      unless bgk == nil
        if freigabe == 1
          geheim = 0
        else
          geheim = 1
        end
      else
        geheim = 0
      end
      
      if geheim == 0
        parentcompany = b.bname1.to_s + " " + b.bname2.to_s
        if parentcompany.strip == ""
          simple_node_with_content_creator(facilityreport,
                                           "rsm:ParentCompanyName", "-")
          if $export_hinweise
            @log.error("Hinweis für Kennnummer '" + b.kennnr +
                       "'! Für Betriebe ohne Vertraulichkeit muss die Muttergesellschaft angegeben werden. " +
                       "Ersatzwert '-' wurde eingetragen")
            zeile = ['Fehler', 10, "Fehlende Muttergesellschaft", b.land_id, b.jahr, Stnd.now, b.kennnr, betriebsname,
                     "Fehlende Muttergesellschaft", b.id, ''].to_csv(:col_sep => ';', :row_sep => nil)
            @log_table.error(zeile)
          end
        else
          simple_node_with_content_creator(facilityreport,
                                           "rsm:ParentCompanyName", parentcompany.strip)
        end
        
        facilityname = b.name1.to_s + " " + b.name2.to_s
        if facilityname.strip == ""
          simple_node_with_content_creator(facilityreport,
                                           "rsm:FacilityName", "-")
          
          @log.error("Hinweis für Kennnummer '" + b.kennnr +
                     "'! Für Betriebe ohne Vertraulichkeit muss der Name der Betriebseinrichtung angegeben werden." +
                     "Ersatzwert '-' wurde eingetragen") if $export_hinweise
          zeile = ['Fehler', 11, "Fehlender Name der Betriebseinrichtung", b.land_id, b.jahr, Stnd.now, b.kennnr, betriebsname,
                   "Fehlender Name der Betriebseinrichtung", b.id, ''].to_csv(:col_sep => ';', :row_sep => nil)
          @log_table.error(zeile)
        else
          simple_node_with_content_creator(facilityreport,
                                           "rsm:FacilityName", facilityname.strip)
        end
        
        adress = simple_node_creator(facilityreport, "rsm:Address")
        
        if b.strasse?
          simple_node_with_content_creator(adress, "rsm:StreetName", b.strasse)
        else
          simple_node_with_content_creator(adress, "rsm:StreetName", "-")
          
          @log.error("Hinweis für Kennnummer '" + b.kennnr +
                     "'! Für Betriebe ohne Vertraulichkeit muss die Straße der Betriebseinrichtung angegeben werden." +
                     "Ersatzwert '-' wurde eingetragen") if $export_hinweise
          zeile = ['Fehler', 12, "Fehlende Straße der Betriebseinrichtung", b.land_id, b.jahr, Stnd.now, b.kennnr, betriebsname,
                   "Fehlende Straße der Betriebseinrichtung", b.id, ''].to_csv(:col_sep => ';', :row_sep => nil)
          @log_table.error(zeile)
        end
        
        if b.nr?
          simple_node_with_content_creator(adress, "rsm:BuildingNumber", b.nr)
        else
          simple_node_with_content_creator(adress, "rsm:BuildingNumber", "-")
          
          @log.error("Hinweis für Kennnummer '" + b.kennnr +
                     "'! Für Betriebe ohne Vertraulichkeit muss die Hausnummer der Betriebseinrichtung angegeben werden." +
                     "Ersatzwert '-' wurde eingetragen") if $export_hinweise
          zeile = ['Fehler', 13, "Fehlende Hausnummer der Betriebseinrichtung", b.land_id, b.jahr, Stnd.now, b.kennnr, betriebsname,
                   "Fehlende Hausnummer der Betriebseinrichtung", b.id, ''].to_csv(:col_sep => ';', :row_sep => nil)
          @log_table.error(zeile)
        end
        
        if b.ort?
          simple_node_with_content_creator(adress, "rsm:CityName", b.ort)
        else
          simple_node_with_content_creator(adress, "rsm:CityName", "-")
          
          @log.error("Hinweis für Kennnummer '" + b.kennnr +
                     "'! Für Betriebe ohne Vertraulichkeit muss der Ort der Betriebseinrichtung angegeben werden." +
                     "Ersatzwert '-' wurde eingetragen") if $export_hinweise
          zeile = ['Fehler', 14, "Fehlender Ort der Betriebseinrichtung", b.land_id, b.jahr, Stnd.now, b.kennnr, betriebsname,
                   "Fehlender Ort der Betriebseinrichtung", b.id, ''].to_csv(:col_sep => ';', :row_sep => nil)
          @log_table.error(zeile)
        end
        
        if b.plzstr?
          simple_node_with_content_creator(adress, "rsm:PostcodeCode", b.plzstr)
        else
          simple_node_with_content_creator(adress, "rsm:PostcodeCode", "-")
          
          @log.error("Hinweis für Kennnummer '" + b.kennnr +
                     "'! Für Betriebe ohne Vertraulichkeit muss die Postleitzahl der Betriebseinrichtung angegeben werden." +
                     "Ersatzwert '-' wurde eingetragen") if $export_hinweise
          zeile = ['Fehler', 15, "Fehlende Postleitzahl der Betriebseinrichtung", b.land_id, b.jahr, Stnd.now, b.kennnr, betriebsname,
                   "Fehlende Postleitzahl der Betriebseinrichtung", b.id, ''].to_csv(:col_sep => ';', :row_sep => nil)
          @log_table.error(zeile)
        end
      end
      
      geographicalcoordinate = simple_node_creator(facilityreport, "rsm:GeographicalCoordinate")
      if geheim == 1
        simple_node_with_content_creator(geographicalcoordinate,
                                         "rsm:LongitudeMeasure", "0.0")
        simple_node_with_content_creator(geographicalcoordinate,
                                         "rsm:LatitudeMeasure", "0.0")
      else
        if b.laengengrad? and b.breitengrad?
          simple_node_with_content_creator(geographicalcoordinate,
                                           "rsm:LongitudeMeasure", b.laengengrad.to_s)
          simple_node_with_content_creator(geographicalcoordinate,
                                           "rsm:LatitudeMeasure", b.breitengrad.to_s)
        else
          @log.error("Fehler in Kennnummer '" + b.kennnr +
                     "'! Längen- und Breitengrad müssen korrekt angegeben werden. (RW/HW=" + b.rechtsw.to_s + "/"+ b.hochw.to_s + ")")
          zeile = ['Fehler', 33, "Fehlende geografische Koordinate", b.land_id, b.jahr, Stnd.now, b.kennnr, betriebsname,
                   "Fehlende geografische Koordinate", b.id, ''].to_csv(:col_sep => ';', :row_sep => nil)
          @log_table.error(zeile)
        end
      end
      
      if b.flezgeb_id?
        f = Flusseinzugsgebiet.find(b.flezgeb_id)
        
        unless f == nil
          translate_de2eu("RiverBasinDistrictID", f.schl, facilityreport,
                          "rsm:RiverBasinDistrictID", b.kennnr, @log)
        end
      else
        @log.error("Fehler in Kennnummer '" + b.kennnr +
                   "'! Das Flusseinzugsgebiet muss angegeben werden.")
        zeile = ['Fehler', 32, "Fehlendes Flusseinzugsgebiet", b.land_id, b.jahr, Stnd.now, b.kennnr, betriebsname,
                 "Fehlendes Flusseinzugsgebiet", b.id, ''].to_csv(:col_sep => ';', :row_sep => nil)
        @log_table.error(zeile)
      end
      
      if b.nace_id?
        n = NaceCode.find(b.nace_id)
        unless n == nil
          translate_de2eu2("NACEMainEconomicActivityCode", n.schl, facilityreport,
                           "rsm:NACEMainEconomicActivityCode","rsm:MainEconomicActivityName", b.kennnr, @log)
        end
      else
        @log.error("Fehler in Kennnummer '" + b.kennnr +
                   "'! Der NACE-Code muss angegeben werden.")
        zeile = ['Fehler', 31, "Fehlender NACE-Code", b.land_id, b.jahr, Stnd.now, b.kennnr, betriebsname,
                 "Fehlender NACE-Code", b.id, ''].to_csv(:col_sep => ';', :row_sep => nil)
        @log_table.error(zeile)
      end
      
      simple_node_with_content_creator(facilityreport,
                                       "rsm:CompetentAuthorityPartyName", @name)
      
      if b.prodvol?
        pv = simple_node_creator(facilityreport, "rsm:ProductionVolume")
        simple_node_with_content_creator(pv,"rsm:ProductName", b.prodvol.to_s)
        simple_node_with_attribut_and_content_creator(pv, "rsm:Quantity", "unitCode", "PCS", "1")
      end
      
      simple_node_with_content_creator(facilityreport,
                                       "rsm:TotalIPPCInstallationQuantity", b.anlzahl.to_s) if b.anlzahl?
      simple_node_with_content_creator(facilityreport,
                                       "rsm:OperationHours", b.betriebh.to_s) if b.betriebh?
      simple_node_with_content_creator(facilityreport,
                                       "rsm:TotalEmployeeQuantity", b.beszahl.to_s) if b.beszahl?
      
      # NutsRegionID könnte hier hin, wenn wir sie hätten (mlt, 2011-03-22)
      
      # rsm:PublicInformation
      simple_node_with_content_creator(facilityreport, "rsm:PublicInformation", b.prtrinfo) if b.prtrinfo?
      
      if geheim == 1 then
        unless bgk == nil
          simple_node_with_content_creator(facilityreport,
                                           "rsm:ConfidentialIndicator", "true")
          
          vv = VertraulichkeitsV.find(bgk.geheimk_id)
          unless vv == nil
            translate_de2eu("ConfidentialCode", vv.schl, facilityreport,
                            "rsm:ConfidentialCode", b.kennnr, @log)
          end
        else
          simple_node_with_content_creator(facilityreport,
                                           "rsm:ConfidentialIndicator", "false")
        end
      else
        simple_node_with_content_creator(facilityreport,
                                         "rsm:ConfidentialIndicator", "false")
      end
      
      simple_node_with_content_creator(facilityreport,
                                       "rsm:ProtectVoluntaryData", "true")
      simple_node_with_content_creator(facilityreport,
                                       "rsm:RemarkText", b.bemerk) if b.bemerk?

      if geheim && betriebsname == ''
        export_taetigkeiten( b, facilityreport, freigabe, 'vertraulich' )
        export_freisetzungen( b, facilityreport, freigabe, 'vertraulich' )
        export_verbringungen( b, facilityreport, freigabe, 'vertraulich' )
        export_abfaelle( b, facilityreport, freigabe, 'vertraulich' )
      else
        export_taetigkeiten( b, facilityreport, freigabe, betriebsname )
        export_freisetzungen( b, facilityreport, freigabe, betriebsname )
        export_verbringungen( b, facilityreport, freigabe, betriebsname )
        export_abfaelle( b, facilityreport, freigabe, betriebsname )
      end
    end # export_betrieb_einzeln
    

    def export_taetigkeiten( b, facilityreport, freigabe, betriebsname )
      taetigkeiten = Taetigkeit.where("state = 0 and p_betrieb_id = ?", b.id).
                     order("htprtr desc, nrprtr_id")
      
      if taetigkeiten.length > 0
        ranknr = 2
        haupttaetigkeit_gefunden = false
        taetigkeiten.each do |taet|
          
          tgk = TaetigkeitGK.where("state = 0 and p_taet_id = ?", taet.id).first
          if tgk == nil || freigabe == 1
            if taet.nrprtr_id?
              prtr = PRTRTaetigkeit.find(taet.nrprtr_id)
              unless prtr == nil
                activity = simple_node_creator(facilityreport, "rsm:Activity")
                if taet.htprtr == 1
                  haupttaetigkeit_gefunden = true
                  simple_node_with_content_creator(activity, "rsm:RankingNumeric", "1")
                else
                  simple_node_with_content_creator(activity, "rsm:RankingNumeric", ranknr.to_s)
                  ranknr += 1
                end
                translate_de2eu("AnnexIActivityCode", prtr.schl, activity,
                                "rsm:AnnexIActivityCode", b.kennnr, @log)
              else
                @log.error("Datenbankfehler in Kennnummer '" + b.kennnr +
                           "'! Konnte ID '" + taet.nrprtr_id.to_s + "' nicht in PRTRTaetigkeit übersetzen.")
                zeile = ['Fehler', 1, "Anwendungsfehler", b.land_id, b.jahr, Stnd.now, b.kennnr, betriebsname,
                         "Anwendungsfehler: PRTR-Tätigkeit #{taet.nrprtr_id} kann nicht übersetzt werden", b.id, ''].to_csv(:col_sep => ';', :row_sep => nil)
                @log_table.error(zeile)
              end
            end
          end
        end
        if !haupttaetigkeit_gefunden
          @log.error("Fehler in Kennnummer '" + b.kennnr +
                     "'! Dieser Betrieb hat ausschließlich Nebentätigkeiten oder eine vertrauliche Haupttätigkeit!")
          zeile = ['Fehler', 92, "Fehlende Haupttätigkeit", b.land_id, b.jahr, Stnd.now, b.kennnr, betriebsname,
                   "Fehlende Haupttätigkeit", b.id, ''].to_csv(:col_sep => ';', :row_sep => nil)
          @log_table.error(zeile)
        end
      else
        @log.error("Fehler in Kennnummer '" + b.kennnr +
                   "'! Für den Betrieb muss mindestens eine Haupttätigkeit angegeben werden.")
        zeile = ['Fehler', 91, "Fehlende Haupttätigkeit (keinerlei Tätigkeiten)", b.land_id, b.jahr, Stnd.now, b.kennnr, betriebsname,
                 "Fehlende Haupttätigkeit (keinerlei Tätigkeiten)", b.id, ''].to_csv(:col_sep => ';', :row_sep => nil)
        @log_table.error(zeile)
      end
    end # export_taetigkeiten
    
    
    def export_freisetzungen( b, facilityreport, freigabe, betriebsname )
      freisetzungen = Freisetzung.where("state = 0 and p_betrieb_id = ?", b.id).order("sgruppe_id")
      if freisetzungen.length > 0
        co2_freis = nil
        tmp_freis = nil
        freisetzungen.each do |freis|
          if freis.gk
            if tmp_freis
              if tmp_freis.sgruppe_id == freis.sgruppe_id
                tmp_freis.aggregate(freis)
                puts "Habe Freisetzungen aggregiert für Betrieb #{b.kennnr}"
                next
              else
                export_freisetzung_einzeln( tmp_freis, b, facilityreport, freigabe, betriebsname, 'NORMAL_MODE' )
                tmp_freis = freis
              end
            else
              tmp_freis = freis
              next
            end
          elsif tmp_freis
            export_freisetzung_einzeln( tmp_freis, b, facilityreport, freigabe, betriebsname, 'NORMAL_MODE' )
            tmp_freis = nil
          end
          my_freis = export_freisetzung_einzeln( freis, b, facilityreport, freigabe, betriebsname, 'NORMAL_MODE' )
          co2_freis = my_freis if my_freis
        end
        if co2_freis && @year >= 2009
          export_freisetzung_einzeln( co2_freis, b, facilityreport, freigabe, betriebsname, 'NON_BIOGENICAL_MODE' )
        end
      end # Ende der Freisetzungen
    end # export_freisetzungen
    
    
    def export_freisetzung_einzeln( freis, b, facilityreport, freigabe, betriebsname, mode )
      if mode == 'NORMAL_MODE'
        co2_freis = nil
        co2_freis = freis if freis.stoffnr_id == 3 && freis.nbiofracht && freis.nbiofracht < freis.jfracht
      end
      release = simple_node_creator(facilityreport, "rsm:PollutantRelease")
      
      fgk = FreisetzungGK.where("state = 0 and p_freis_id = ?", freis.id).first
      
      medium = Medium.find(freis.medium_id)
      if medium
        translate_de2eu("MediumCode", medium.schl, release,
                        "rsm:MediumCode", b.kennnr, @log)
      else
        @log.error("Fehler in Kennnummer '" + b.kennnr + "', Freisetzungs-ID '" + freis.id.to_s +
                   "'! Hier fehlt das Medium (Umweltkompartiment)!")
        zeile = ['Fehler', 100, "Fehlendes Medium (Umweltkompartiment) bei Freisetzung",
                 b.land_id, b.jahr, Stnd.now, b.kennnr, betriebsname, "Fehlendes Medium (Umweltkompartiment) bei Freisetzung",
                 b.id, freis.id].to_csv(:col_sep => ';', :row_sep => nil)
        @log_table.error(zeile)
      end
      
      if fgk == nil || freigabe == 1 # keine Vertraulichkeit
        if mode == 'NON_BIOGENICAL_MODE'
          pollutant = Stoff.find(92)
        else
          pollutant = Stoff.find(freis.stoffnr_id)
        end
        if pollutant
          translate_de2eu("PollutantCode", pollutant.schl, release,
                          "rsm:PollutantCode", b.kennnr, @log)
        elsif fgk
          # Offenbar ist Freigabe erteilt worden. Trotzdem fehlt der Stoff:
          @log.error("Fehler in Kennnummer '" + b.kennnr + "', Freisetzungs-ID '" + freis.id.to_s +
                     "'! Hier fehlt der Stoff, den vermutlich der XML1-auf-XML2-Konverter entfernt hat.")
          fehlernummer = freis.medium_id * 100 + 5
          zeile = ['Fehler', fehlernummer, "Fehlender Stoff bei Freisetzung #{medium.ktext}",
                   b.land_id, b.jahr, Stnd.now, b.kennnr, betriebsname, "Fehlender Stoff bei Freisetzung #{medium.ktext} (Vertraulichkeit, aber Freigabe)",
                   b.id, freis.id].to_csv(:col_sep => ';', :row_sep => nil)
          @log_table.error(zeile)
        else
          @log.error("Fehler in Kennnummer '" + b.kennnr + "', Freisetzungs-ID '" + freis.id.to_s +
                     "'! Hier fehlt der Stoff!")
          fehlernummer = freis.medium_id * 100 + 5
          zeile = ['Fehler', fehlernummer, "Fehlender Stoff bei Freisetzung #{medium.ktext}",
                   b.land_id, b.jahr, Stnd.now, b.kennnr, betriebsname, "Fehlender Stoff bei Freisetzung #{medium.ktext}",
                   b.id, freis.id].to_csv(:col_sep => ';', :row_sep => nil)
          @log_table.error(zeile)
        end
      else
        pollutant = Schadstoffgruppe.find(freis.sgruppe_id)
        if pollutant
          translate_de2eu("PollutantCode", pollutant.schl, release,
                          "rsm:PollutantCode", b.kennnr, @log)
        else
          @log.error("Fehler in Kennnummer '" + b.kennnr + "', Freisetzungs-ID '" + freis.id.to_s +
                     "'! Hier fehlt die Stoffgruppe!")
          fehlernummer = freis.medium_id * 100 + 6
          zeile = ['Fehler', fehlernummer, "Fehlende Stoffgruppe bei Freisetzung #{medium.ktext} (Vertraulichkeit)",
                   b.land_id, b.jahr, Stnd.now, b.kennnr, betriebsname, "Fehlende Stoffgruppe bei Freisetzung #{medium.ktext} (Vertraulichkeit)",
                   b.id, freis.id].to_csv(:col_sep => ';', :row_sep => nil)
          @log_table.error(zeile)
        end
      end
      
      methodbasis = nil
      methodbasis = Bestimmungsmethode.find(freis.bestim_id) if freis.bestim_id
      if methodbasis
        simple_node_with_content_creator(release,
                                         "rsm:MethodBasisCode", methodbasis.schl)
      else
        @log.error("Fehler in Kennnummer '" + b.kennnr + "', Freisetzungs-ID '" + freis.id.to_s +
                   "'! Hier fehlt die Ermittlungsart (Bestimmungsmethode (C/E/M)!")
        fehlernummer = freis.medium_id * 100 + 7
        zeile = ['Fehler', fehlernummer, "Fehlende Ermittlungsart bei Freisetzung #{medium.ktext} (CEM)",
                 b.land_id, b.jahr, Stnd.now, b.kennnr, betriebsname, "Fehlende Ermittlungsart bei Freisetzung #{medium.ktext} (CEM)",
                 b.id, freis.id].to_csv(:col_sep => ';', :row_sep => nil)
        @log_table.error(zeile)
      end
      
      if fgk == nil || freigabe == 1 # keine Vertraulichkeit
        if freis.bvcode_id?
          methodeused = Bestimmungsverfahren.find(freis.bvcode_id)
          
          unless methodeused == nil
            method = simple_node_creator(release, "rsm:MethodUsed")
            
            eu_code = translate_de2eu("MethodTypeCode", methodeused.schl, method,
                                      "rsm:MethodTypeCode", b.kennnr, @log)
            
            if freis.messverf?
              designation = freis.messverf.gsub(/\s/, " ")
              simple_node_with_content_creator(method, "rsm:Designation", designation)
              if freis.messverf.length > 255
                #designation = freis.messverf[0, 251] + "..."
                @log.error("Fehler in Kennnummer '" + b.kennnr + "', Freisetzungs-ID '" + freis.id.to_s +
                           "'! Designation ist zu lang! Der Text lautet: " + designation)
                fehlernummer = freis.medium_id * 100 + 3
                zeile = ['Fehler', fehlernummer, "Zu lange Beschreibung (messverf) des Best.verf. Freisetzung #{medium.ktext}",
                         b.land_id, b.jahr, Stnd.now, b.kennnr, betriebsname,
                         "#{freis.bvcode_id}, #{methodeused.schl}: Zu lange Beschreibung (messverf) des Best.verf. Freisetzung #{medium.ktext}: #{freis.messverf}",
                         b.id, freis.id].to_csv(:col_sep => ';', :row_sep => nil)
                @log_table.error(zeile)
                @log_designation.error("Fehler in Kennnummer '" + b.kennnr + "', Name1 '" + b.name1.to_s +
                                       "', Freisetzungs-ID '" + freis.id.to_s +
                                       "'! Designation ist zu lang! Der Text lautet: " + designation)
              end
            else
              case eu_code
              when "CEN/ISO", "IPCC", "ETS", "UNECE/EMEP"
                @log.error("Fehler in Kennnummer '" + b.kennnr + "', Freisetzungs-ID '" + freis.id.to_s +
                           "'! Hier fehlt die Beschreibung des Bestimmungsverfahrens! (bvcode ist '" + methodeused.schl + "' gewesen.)")
                fehlernummer = freis.medium_id * 100 + 2
                zeile = ['Fehler', fehlernummer, "#{freis.bvcode_id}, #{methodeused.schl}: Fehlende Beschreibung (messverf) des Best.verf. Freisetzung #{medium.ktext}",
                         b.land_id, b.jahr, Stnd.now, b.kennnr, betriebsname, "Fehlende Beschreibung (messverf) des Best.verf. Freisetzung #{medium.ktext}",
                         b.id, freis.id].to_csv(:col_sep => ';', :row_sep => nil)
                @log_table.error(zeile)
              end
            end
          else
            @log.error("Datenbankfehler in Kennnummer '" + b.kennnr + "', Freisetzungs-ID '" + freis.id.to_s +
                       "'! Bestimmungsverfahren (bvcode_id) '" + freis.bvcode_id.to_s + "' konnte nicht übersetzt werden!")
            zeile = ['Fehler', 1, "Anwendungsfehler",
                     b.land_id, b.jahr, Stnd.now, b.kennnr, betriebsname,
                     "Freisetzung #{medium.ktext}, bvcode_id = #{freis.bvcode_id} kann nicht übersetzt werden",
                     b.id, freis.id].to_csv(:col_sep => ';', :row_sep => nil)
            @log_table.error(zeile)
          end
        else
          if !(methodbasis && methodbasis.schl == "E")
            @log.error("Fehler in Kennnummer '" + b.kennnr + "', Freisetzungs-ID '" + freis.id.to_s +
                       "'! Hier fehlt das Bestimmungsverfahren!")
            fehlernummer = freis.medium_id * 100 + 1
            zeile = ['Fehler', fehlernummer, "Fehlendes Bestimmungsverfahren (bvcode) Freisetzung #{medium.ktext}",
                     b.land_id, b.jahr, Stnd.now, b.kennnr, betriebsname, "Fehlendes Bestimmungsverfahren (bvcode) Freisetzung #{medium.ktext} und Ermittlungsmethode ist nicht E (geschätzt)",
                     b.id, freis.id].to_csv(:col_sep => ';', :row_sep => nil)
            @log_table.error(zeile)
          end
        end
      end
      
      if mode == 'NORMAL_MODE'
        if freis.jfracht?
          simple_node_with_content_creator(release, "rsm:TotalQuantity", freis.jfracht.to_3signdigits)
        else
          @log.error("Fehler in Kennnummer '" + b.kennnr +
                     "'! Bei 'Freisetzung' muss die Jahresfracht angegeben werden.")
          # Fehlertabellenzeile erstellen
          fehlernummer = freis.medium_id * 100 + 4
          zeile = ['Fehler', fehlernummer, "Fehlende Jahresfracht Freisetzung #{medium.ktext}",
                   b.land_id, b.jahr, Stnd.now, b.kennnr, betriebsname, "Fehlende Jahresfracht bei Freisetzung #{medium.ktext}",
                   b.id, freis.id].to_csv(:col_sep => ';', :row_sep => nil)
          @log_table.error(zeile)
        end
      else
        # NON_BIOGENICAL_MODE:
        simple_node_with_content_creator(release, "rsm:TotalQuantity", freis.nbiofracht.to_3signdigits)
      end
      
      if freis.vfracht? && mode == 'NORMAL_MODE'
        simple_node_with_content_creator(release, "rsm:AccidentalQuantity", freis.vfracht.to_3signdigits)
      else
        simple_node_with_content_creator(release, "rsm:AccidentalQuantity", "0.00")
        #@log.error("Warnung für Kennnummer '" + b.kennnr +
        #    "'! Bei 'Freisetzung' wurde 'Fracht versehentlich' durch das Programm auf 0 gesetzt!")
      end
      
      if fgk == nil || freigabe == 1 # Vertraulichkeit
        simple_node_with_content_creator(release, "rsm:ConfidentialIndicator", "false")
      else
        simple_node_with_content_creator(release, "rsm:ConfidentialIndicator", "true")
        vv = VertraulichkeitsV.find(fgk.geheimk_id)
        unless vv == nil
          translate_de2eu("ConfidentialCode", vv.schl, release,
                          "rsm:ConfidentialCode", b.kennnr, @log)
          simple_node_with_content_creator(release, "rsm:RemarkText", freis.geheimd) if freis.geheimd?
        end
      end
      return co2_freis
    end # export_freisetzung_einzeln
    
    
    def export_verbringungen( b, facilityreport, freigabe, betriebsname )
      vabwasser = VAbwasser.where("state = 0 and p_betrieb_id = ?", b.id).order("sgruppe_id")
      
      if vabwasser.length > 0
        tmp_vabw = nil
        vabwasser.each do |vabw|
          if vabw.gk
            if tmp_vabw
              if tmp_vabw.sgruppe_id == vabw.sgruppe_id
                tmp_vabw.aggregate(vabw)
                puts "Habe Verbringungen aggregiert für Betrieb #{b.kennnr}"
                next
              else
                export_verbringungen_einzeln( tmp_vabw, b, facilityreport, freigabe, betriebsname)
                tmp_vabw = vabw
              end
            else
              tmp_vabw = vabw
              next
            end
          elsif tmp_vabw
            export_verbringungen_einzeln( tmp_vabw, b, facilityreport, freigabe, betriebsname)
            tmp_vabw = nil
          end
          export_verbringungen_einzeln(vabw, b, facilityreport, freigabe, betriebsname)
        end
      end # if vabwasser
    end # export_verbringungen

    def export_verbringungen_einzeln(vabw, b,facilityreport, freigabe, betriebsname )
                transfer = simple_node_creator(facilityreport, "rsm:PollutantTransfer")

          vagk = VAbwasserGK.where("state = 0 and p_vabw_id = ?", vabw.id).first

          if vagk == nil || freigabe == 1 # keine Vertraulichkeit
            pollutant = Stoff.find(vabw.stoffnr_id)
            if pollutant
              translate_de2eu("PollutantCode", pollutant.schl, transfer,
                              "rsm:PollutantCode", b.kennnr, @log)
            else
              @log.error("Fehler in Kennnummer '" + b.kennnr + "', Abwasser-ID '" + vabw.id.to_s +
                         "'! Hier fehlt der Stoff!")
              zeile = ['Fehler', 405, "Fehlender Stoff bei Abwasser",
                       b.land_id, b.jahr, Stnd.now, b.kennnr, betriebsname, "Fehlender Stoff bei Abwasser",
                       b.id, vabw.id].to_csv(:col_sep => ';', :row_sep => nil)
              @log_table.error(zeile)
            end
          else
            pollutant = Schadstoffgruppe.find(vabw.sgruppe_id)
            if pollutant
              translate_de2eu("PollutantCode", pollutant.schl, transfer,
                              "rsm:PollutantCode", b.kennnr, @log)
            else
              @log.error("Fehler in Kennnummer '" + b.kennnr + "', Abwasser-ID '" + vabw.id.to_s +
                         "'! Hier fehlt die Stoffgruppe!")
              zeile = ['Fehler', 406, "Fehlende Stoffgruppe bei Abwasser (Vertraulichkeit)",
                       b.land_id, b.jahr, Stnd.now, b.kennnr, betriebsname, "Fehlende Stoffgruppe bei Abwasser (Vertraulichkeit)",
                       b.id, vabw.id].to_csv(:col_sep => ';', :row_sep => nil)
              @log_table.error(zeile)
            end
          end

          if vabw.bestim_id?
            methodbasis = Bestimmungsmethode.find(vabw.bestim_id)
            if methodbasis
              simple_node_with_content_creator(transfer,
                                               "rsm:MethodBasisCode", methodbasis.schl)
            else
              @log.error("Fehler in Kennnummer '" + b.kennnr + "', Abwasser-ID '" + vabw.id.to_s +
                         "'! Hier fehlt die Bestimmungsmethode (M/C/E)!")
              zeile = ['Fehler', 407, "Fehlende Ermittlungsart Abwasser (CEM)",
                       b.land_id, b.jahr, Stnd.now, b.kennnr, betriebsname, "Fehlende Ermittlungsart Abwasser (CEM)",
                       b.id, vabw.id].to_csv(:col_sep => ';', :row_sep => nil)
              @log_table.error(zeile)
            end
          end

          if vagk == nil || freigabe == 1 # keine Vertraulichkeit
            if vabw.bvcode_id?
              methodeused = Bestimmungsverfahren.find(vabw.bvcode_id)

              unless methodeused == nil
                method = simple_node_creator(transfer, "rsm:MethodUsed")

                eu_code = translate_de2eu("MethodTypeCode", methodeused.schl, method,
                                          "rsm:MethodTypeCode", b.kennnr, @log)

                if vabw.messverf?
                  designation = vabw.messverf.gsub(/\s/, " ")
                  simple_node_with_content_creator(method, "rsm:Designation", designation)

                  if vabw.messverf.length > 255
                    #designation = vabw.messverf[0, 251] + "..."
                    @log.error("Fehler in Kennnummer '" + b.kennnr + "', Abwasser-ID '" + vabw.id.to_s +
                               "'! Designation ist zu lang! Der Text lautet: " + designation)
                    zeile = ['Fehler', 403, "Zu lange Beschreibung (messverf) des Best.verf. Abwasser",
                             b.land_id, b.jahr, Stnd.now, b.kennnr, betriebsname,
                             "#{vabw.bvcode_id}, #{methodeused.schl}: Zu lange Beschreibung (messverf) des Best.verf. Abwasser: #{vabw.messverf}",
                             b.id, vabw.id].to_csv(:col_sep => ';', :row_sep => nil)
                    @log_table.error(zeile)
                    @log_designation.error("Fehler in Kennnummer '" + b.kennnr + "', Name1 '" + b.name1.to_s +
                                           "', Abwasser-ID '" + vabw.id.to_s +
                                           "'! Designation ist zu lang! Der Text lautet: " + designation)
                  end

                else
                  case eu_code
                  when "CEN/ISO", "IPCC", "ETS", "UNECE/EMEP"
                    @log.error("Fehler in Kennnummer '" + b.kennnr + "', Abwasser-ID '" + vabw.id.to_s +
                               "'! Hier fehlt die Beschreibung des Bestimmungsverfahrens! (bvcode ist '" + methodeused.schl + "' gewesen.)")
                    zeile = ['Fehler', 402, "Fehlende Beschreibung (messverf) des Best.verf. Abwasser",
                             b.land_id, b.jahr, Stnd.now, b.kennnr, betriebsname,
                             "#{vabw.bvcode_id}, #{methodeused.schl}: Fehlende Beschreibung (messverf) des Best.verf. Abwasser",
                             b.id, vabw.id].to_csv(:col_sep => ';', :row_sep => nil)
                    @log_table.error(zeile)
                  end
                end
              else
                @log.error("Datenbankfehler in Kennnummer '" + b.kennnr + "', Abwasser-ID '" + vabw.id.to_s +
                           "'! Bestimmungsverfahren (bvcode_id) '" + vabw.bvcode_id.to_s + "' konnte nicht übersetzt werden!")
                zeile = ['Fehler', 1, "Anwendungsfehler",
                         b.land_id, b.jahr, Stnd.now, b.kennnr, betriebsname,
                         "Abwasser, bvcode_id = #{vabw.bvcode_id} kann nicht übersetzt werden",
                         b.id, vabw.id].to_csv(:col_sep => ';', :row_sep => nil)
                @log_table.error(zeile)
              end
            else
              if !(methodbasis && methodbasis.schl == "E")
                @log.error("Fehler in Kennnummer '" + b.kennnr + "', Abwasser-ID '" + vabw.id.to_s +
                           "'! Hier fehlt das Bestimmungsverfahren!")
                zeile = ['Fehler', 401, "Fehlendes Bestimmungsverfahren (bvcode) Abwasser",
                         b.land_id, b.jahr, Stnd.now, b.kennnr, betriebsname, "Fehlendes Bestimmungsverfahren (bvcode) Abwasser",
                         b.id, vabw.id].to_csv(:col_sep => ';', :row_sep => nil)
                @log_table.error(zeile)
              end
            end
          end

          if vabw.jfracht?
            simple_node_with_content_creator(transfer,
                                             "rsm:Quantity", vabw.jfracht.to_3signdigits)
          else
            @log.error("Fehler in Kennnummer '" + b.kennnr + "', Abwasser-ID '" + vabw.id.to_s +
                       "'! Bei 'Verbringung Abwasser' muss die Jahresfracht angegeben werden.")
            # Fehlertabellenzeile erstellen
            fehlernummer = 404
            zeile = ['Fehler', fehlernummer, "Fehlende Jahresfracht bei Abwasser",
                     b.land_id, b.jahr, Stnd.now, b.kennnr, betriebsname, "Fehlende Jahresfracht bei Abwasser",
                     b.id, vabw.id].to_csv(:col_sep => ';', :row_sep => nil)
            @log_table.error(zeile)

          end

          unless vagk == nil || freigabe == 1 # Vertraulichkeit
            simple_node_with_content_creator(transfer, "rsm:ConfidentialIndicator", "true")

            vv = VertraulichkeitsV.find(vagk.geheimk_id)
            unless vv == nil
              translate_de2eu("ConfidentialCode", vv.schl, transfer,
                              "rsm:ConfidentialCode", b.kennnr, @log)
              simple_node_with_content_creator(transfer, "rsm:RemarkText", vabw.geheimd) if vabw.geheimd?
            end
          else
            simple_node_with_content_creator(transfer, "rsm:ConfidentialIndicator", "false")
          end
    end# export_verbringungen_einzeln
    
    def export_abfaelle( b, facilityreport, freigabe, betriebsname )
      vabfall = VAbfall.where("state = 0 and p_betrieb_id = ?", b.id)
      
      if vabfall.length > 0
        
        vabfall.each do |vabf|
          transfer = simple_node_creator(facilityreport, "rsm:WasteTransfer")
          
          if vabf.abfart_id == 2 # Nicht gefährlicher Abfall
            wastetypecode = "NON-HW"
          else
            # Gefährlicher Abfall
            if vabf.aland_id?
              if vabf.aland_id == 2 # innerhalb
                wastetypecode = "HWIC"
              else
                wastetypecode = "HWOC" # außerhalb
              end
            else
              # Entscheidung Frau Grimm: Wenn nicht bekannt, dann Inland (ehemals: VOID):
              # Neue Entscheidung Frau Grimm (siehe Mail an rmi 05.04.2012 09:16): Werden sowieso unterdrückt und sind nur 2 Betriebe aus 2007, also nur Fehler ausgeben
              wastetypecode = "HW"
              zeile = ['Fehler', 508, "Fehlende Angabe Inland/Ausland bei gefährlichem Abfall (HW)",
                       b.land_id, b.jahr, Stnd.now, b.kennnr, betriebsname, "Fehlende Angabe Inland/Ausland bei gefährlichem Abfall (HW)",
                       b.id, vabf.id].to_csv(:col_sep => ';', :row_sep => nil)
              @log_table.error(zeile)
            end
          end
          
          simple_node_with_content_creator(transfer, "rsm:WasteTypeCode", wastetypecode)
          
          vagk = VAbfallGK.where("state = 0 and p_vabf_id = ?", vabf.id).first
          
          if vagk == nil || freigabe == 1
            # Nicht geheim: ==============================================================
            
            if vabf.verbes_id?
              if vabf.verbes_id == 2
                wastetreatmentcode = "R" # Zur Verwertung
              else vabf.verbes_id == 1
                wastetreatmentcode = "D" # Zur Beseitigung
              end
            else
              @log.error("Fehler in Kennnummer '" + b.kennnr + "', Abfall-ID '" + vabf.id.to_s +
                         "'! Bei 'Verbringung Abfall' muss Abfalldisposition (Verwert./Beseit.) angegeben werden.")
              zeile = ['Fehler', 517, "Fehlende Abfalldisposition (1-Beseitigung / 2-Verwertung)",
                       b.land_id, b.jahr, Stnd.now, b.kennnr, betriebsname, "Fehlende Abfalldisposition (1-Beseitigung / 2-Verwertung) (#{wastetypecode})",
                       b.id, vabf.id].to_csv(:col_sep => ';', :row_sep => nil)
              @log_table.error(zeile)
            end
            
            simple_node_with_content_creator(transfer,
                                             "rsm:WasteTreatmentCode", wastetreatmentcode)
            
            if vabf.menge?
              simple_node_with_content_creator(transfer,
                                               "rsm:Quantity", vabf.menge.to_3signdigits)
            else
              @log.error("Fehler in Kennnummer '" + b.kennnr + "', Abfall-ID '" + vabf.id.to_s +
                         "'! Bei 'Verbringung Abfall' muss die Abfallmenge angegeben werden.")
              zeile = ['Fehler', 518, "Fehlende Abfallmenge",
                       b.land_id, b.jahr, Stnd.now, b.kennnr, betriebsname, "Fehlende Abfallmenge (#{wastetypecode}, #{wastetreatmentcode})",
                       b.id, vabf.id].to_csv(:col_sep => ';', :row_sep => nil)
              @log_table.error(zeile)
            end
            
            if vabf.bestim_id?
              methodbasis = Bestimmungsmethode.find(vabf.bestim_id)
              simple_node_with_content_creator(transfer, "rsm:MethodBasisCode", methodbasis.schl)
            else
              @log.error("Fehler in Kennnummer '" + b.kennnr + "', Abfall-ID '" + vabf.id.to_s +
                         "'! Bei 'Verbringung Abfall' muss die Ermittlungsart (M/C/E) angegeben werden.")
              zeile = ['Fehler', 507, "Fehlende Ermittlungsart Abfall (CEM)",
                       b.land_id, b.jahr, Stnd.now, b.kennnr, betriebsname, "Fehlende Ermittlungsart Abfall (CEM) (#{wastetypecode}, #{wastetreatmentcode})",
                       b.id, vabf.id].to_csv(:col_sep => ';', :row_sep => nil)
              @log_table.error(zeile)
            end
            
            if vabf.bvcode_id?
              
              if vabf.bvcode_id == 1 or vabf.bvcode_id == 2 # Sonstige Methode
                methodused = "OTH"
              else
                methodused = "WEIGH"
              end
              method = simple_node_creator(transfer, "rsm:MethodUsed")
              simple_node_with_content_creator(method, "rsm:MethodTypeCode", methodused)
              
              if vabf.messverf?
                designation = vabf.messverf.gsub(/\s/, " ")
                simple_node_with_content_creator(method, "rsm:Designation", designation)
                
                if vabf.messverf.length > 255
                  #designation = vabf.messverf[0, 251] + "..."
                  @log.error("Fehler in Kennnummer '" + b.kennnr + "', Abfall-ID '" + vabf.id.to_s +
                             "'! Designation ist zu lang! Der Text lautet: " + designation)
                  zeile = ['Fehler', 503, "Zu lange Beschreibung (messverf) des Best.verf. Abfall",
                           b.land_id, b.jahr, Stnd.now, b.kennnr, betriebsname,
                           "#{vabf.bvcode_id}, #{methodused}, #{wastetypecode}, #{wastetreatmentcode}: Zu lange Beschreibung (messverf) des Best.verf. Abfall: #{vabf.messverf}",
                           b.id, vabf.id].to_csv(:col_sep => ';', :row_sep => nil)
                  @log_table.error(zeile)
                  @log_designation.error("Fehler in Kennnummer '" + b.kennnr +
                                         "', Name1 '" + b.name1.to_s + "', Abfall-ID '" + vabf.id.to_s +
                                         "'! Designation ist zu lang! Der Text lautet: " + designation)
                end
              end
            else
              if !(methodbasis && methodbasis.schl == "E")
                @log.error("Fehler in Kennnummer '" + b.kennnr + "', Abfall-ID '" + vabf.id.to_s +
                           "'! Hier fehlt das Bestimmungsverfahren!")
                zeile = ['Fehler', 501, "Fehlendes Bestimmungsverfahren (bvcode) in Abfall",
                         b.land_id, b.jahr, Stnd.now, b.kennnr, betriebsname,
                         "Fehlendes Bestimmungsverfahren (bvcode) in Abfall (#{wastetypecode}, #{wastetreatmentcode})",
                         b.id, vabf.id].to_csv(:col_sep => ';', :row_sep => nil)
                @log_table.error(zeile)
              end
            end
            
            simple_node_with_content_creator(transfer,
                                             "rsm:ConfidentialIndicator", "false")
            
            if wastetypecode == "HWOC"
              wastehandlerparty = simple_node_creator(transfer, "rsm:WasteHandlerParty")
              if vabf.vbname?
                simple_node_with_content_creator(wastehandlerparty,
                                                 "rsm:Name", vabf.vbname)
              else
                @log.error("Fehler in Kennnummer '" + b.kennnr + "', Abfall-ID '" + vabf.id.to_s +
                           "'! Hier fehlt der Name des Verwerters/Beseitigers!")
                zeile = ['Fehler', 511, "Fehlender Name des Verwerters/Beseitigers",
                         b.land_id, b.jahr, Stnd.now, b.kennnr, betriebsname,
                         "Fehlender Name des Verwerters/Beseitigers (#{wastetypecode}, #{wastetreatmentcode})",
                         b.id, vabf.id].to_csv(:col_sep => ';', :row_sep => nil)
                @log_table.error(zeile)
              end
              
              address = simple_node_creator(wastehandlerparty, "rsm:Address")
              
              if vabf.vbstr?
                simple_node_with_content_creator(address,
                                                 "rsm:StreetName", vabf.vbstr)
              else
                @log.error("Fehler in Kennnummer '" + b.kennnr + "', Abfall-ID '" + vabf.id.to_s +
                           "'! Hier fehlt die Straße des Verwerters/Beseitigers!")
                zeile = ['Fehler', 512, "Fehlende Straße des Verwerters/Beseitigers",
                         b.land_id, b.jahr, Stnd.now, b.kennnr, betriebsname,
                         "Fehlende Straße des Verwerters/Beseitigers (#{wastetypecode}, #{wastetreatmentcode})",
                         b.id, vabf.id].to_csv(:col_sep => ';', :row_sep => nil)
                @log_table.error(zeile)
              end
              
              if vabf.vbstrnr?
                simple_node_with_content_creator(address,
                                                 "rsm:BuildingNumber", vabf.vbstrnr)
              else
                simple_node_with_content_creator(address,
                                                 "rsm:BuildingNumber", "-")
                @log.error("Fehler in Kennnummer '" + b.kennnr + "', Abfall-ID '" + vabf.id.to_s +
                           "'! Hier fehlt die Hausnummer des Verwerters/Beseitigers!")
                zeile = ['Fehler', 513, "Fehlende Hausnummer des Verwerters/Beseitigers",
                         b.land_id, b.jahr, Stnd.now, b.kennnr, betriebsname,
                         "Fehlende Hausnummer des Verwerters/Beseitigers (#{wastetypecode}, #{wastetreatmentcode})",
                         b.id, vabf.id].to_csv(:col_sep => ';', :row_sep => nil)
                @log_table.error(zeile)
              end
              
              if vabf.vbort?
                simple_node_with_content_creator(address,
                                                 "rsm:CityName", vabf.vbort)
              else
                @log.error("Fehler in Kennnummer '" + b.kennnr + "', Abfall-ID '" + vabf.id.to_s +
                           "'! Hier fehlt der Ort des Verwerters/Beseitigers!")
                zeile = ['Fehler', 514, "Fehlender Ort des Verwerters/Beseitigers",
                         b.land_id, b.jahr, Stnd.now, b.kennnr, betriebsname,
                         "Fehlender Ort des Verwerters/Beseitigers (#{wastetypecode}, #{wastetreatmentcode})",
                         b.id, vabf.id].to_csv(:col_sep => ';', :row_sep => nil)
                @log_table.error(zeile)
              end
              
              if vabf.vbplz?
                simple_node_with_content_creator(address,
                                                 "rsm:PostcodeCode", vabf.vbplz)
              else
                @log.error("Fehler in Kennnummer '" + b.kennnr + "', Abfall-ID '" + vabf.id.to_s +
                           "'! Hier fehlt die Postleitzahl des Verwerters/Beseitigers!")
                zeile = ['Fehler', 515, "Fehlende Postleitzahl des Verwerters/Beseitigers",
                         b.land_id, b.jahr, Stnd.now, b.kennnr, betriebsname,
                         "Fehlende Postleitzahl des Verwerters/Beseitigers (#{wastetypecode}, #{wastetreatmentcode})",
                         b.id, vabf.id].to_csv(:col_sep => ';', :row_sep => nil)
                @log_table.error(zeile)
              end
              
              if @year >= 2008 then
                if vabf.vbstaat_id?
                  vbstaat = StaatISOCode.find(vabf.vbstaat_id)
                  simple_node_with_content_creator(address, "rsm:CountryID", vbstaat.schl)
                else
                  @log.error("Fehler in Kennnummer '" + b.kennnr + "', Abfall-ID '" + vabf.id.to_s +
                             "'! Hier fehlt der Staat (vbstaat) des Verwerters-/Beseitigers!")
                  zeile = ['Fehler', 520, "Fehlender Staat des Verwerters/Beseitigers",
                           b.land_id, b.jahr, Stnd.now, b.kennnr, betriebsname,
                           "Fehlender Staat des Verwerters/Beseitigers (#{wastetypecode}, #{wastetreatmentcode})",
                           b.id, vabf.id].to_csv(:col_sep => ';', :row_sep => nil)
                  @log_table.error(zeile)
                end
                # raise Exception.new, "ERROR. Feature not implemented yet."
              else
                # @year <= 2007:
                if vabf.astaat_id?
                  vbstaat = StaatISOCode.find(vabf.astaat_id)
                  simple_node_with_content_creator(address, "rsm:CountryID", vbstaat.schl)
                elsif vabf.vbstaat_id?
                  vbstaat = StaatISOCode.find(vabf.vbstaat_id)
                  simple_node_with_content_creator(address, "rsm:CountryID", vbstaat.schl)
                else
                  @log.error("Fehler in Kennnummer '" + b.kennnr + "', Abfall-ID '" + vabf.id.to_s +
                             "'! Hier fehlt der Staat (astaat oder vbstaat) des Verwerters-/Beseitigers!")
                  zeile = ['Fehler', 520, "Fehlender Staat des Verwerters/Beseitigers",
                           b.land_id, b.jahr, Stnd.now, b.kennnr, betriebsname,
                           "Fehlender Staat des Verwerters/Beseitigers (astaat, da Jahr < 2008) (#{wastetypecode}, #{wastetreatmentcode})",
                           b.id, vabf.id].to_csv(:col_sep => ';', :row_sep => nil)
                  @log_table.error(zeile)
                end
                # raise Exception.new, "ERROR. Feature not implemented yet."
              end
              
              siteaddress = simple_node_creator(wastehandlerparty, "rsm:SiteAddress")
              
              if vabf.vbsstr?
                simple_node_with_content_creator(siteaddress,
                                                 "rsm:StreetName", vabf.vbsstr)
              else
                @log.error("Fehler in Kennnummer '" + b.kennnr + "', Abfall-ID '" + vabf.id.to_s +
                           "'! Hier fehlt die Straße des Verwertungs-/Beseitigungsstandortes!")
                zeile = ['Fehler', 522, "Fehlende Straße des Verwertungs-/Beseitigungsstandortes",
                         b.land_id, b.jahr, Stnd.now, b.kennnr, betriebsname,
                         "Fehlende Straße des Verwertungs-/Beseitigungsstandortes (#{wastetypecode}, #{wastetreatmentcode})",
                         b.id, vabf.id].to_csv(:col_sep => ';', :row_sep => nil)
                @log_table.error(zeile)
              end
              
              if vabf.vbsstrnr?
                simple_node_with_content_creator(siteaddress,
                                                 "rsm:BuildingNumber", vabf.vbsstrnr)
              else
                @log.error("Fehler in Kennnummer '" + b.kennnr + "', Abfall-ID '" + vabf.id.to_s +
                           "'! Hier fehlt die Hausnummer des Verwertungs-/Beseitigungsstandortes!")
                zeile = ['Fehler', 523, "Fehlende Hausnummer des Verwertungs-/Beseitigungsstandortes",
                         b.land_id, b.jahr, Stnd.now, b.kennnr, betriebsname,
                         "Fehlende Hausnummer des Verwertungs-/Beseitigungsstandortes (#{wastetypecode}, #{wastetreatmentcode})",
                         b.id, vabf.id].to_csv(:col_sep => ';', :row_sep => nil)
                @log_table.error(zeile)
              end
              
              if vabf.vbsort?
                simple_node_with_content_creator(siteaddress,
                                                 "rsm:CityName", vabf.vbsort)
              else
                @log.error("Fehler in Kennnummer '" + b.kennnr + "', Abfall-ID '" + vabf.id.to_s +
                           "'! Hier fehlt der Ort des Verwertungs-/Beseitigungsstandortes!")
                zeile = ['Fehler', 524, "Fehlender Ort des Verwertungs-/Beseitigungsstandortes",
                         b.land_id, b.jahr, Stnd.now, b.kennnr, betriebsname,
                         "Fehlender Ort des Verwertungs-/Beseitigungsstandortes (#{wastetypecode}, #{wastetreatmentcode})",
                         b.id, vabf.id].to_csv(:col_sep => ';', :row_sep => nil)
                @log_table.error(zeile)
              end
              
              if vabf.vbsplz?
                simple_node_with_content_creator(siteaddress,
                                                 "rsm:PostcodeCode", vabf.vbsplz)
              else
                @log.error("Fehler in Kennnummer '" + b.kennnr + "', Abfall-ID '" + vabf.id.to_s +
                           "'! Hier fehlt die Postleitzahl des Verwertungs-/Beseitigungsstandortes!")
                zeile = ['Fehler', 525, "Fehlende Postleitzahl des Verwertungs-/Beseitigungsstandortes",
                         b.land_id, b.jahr, Stnd.now, b.kennnr, betriebsname,
                         "Fehlende Postleitzahl des Verwertungs-/Beseitigungsstandortes (#{wastetypecode}, #{wastetreatmentcode})",
                         b.id, vabf.id].to_csv(:col_sep => ';', :row_sep => nil)
                @log_table.error(zeile)
              end

              # Jetzt doch auch für 2007er Sätze ausgeben:
              if @year >= 2008 || @year <= 2007 then
                if vabf.vbsstaat_id?
                  vbsstaat = StaatISOCode.find(vabf.vbsstaat_id)
                  simple_node_with_content_creator(siteaddress, "rsm:CountryID", vbsstaat.schl)
                else
                  @log.error("Fehler in Kennnummer '" + b.kennnr + "', Abfall-ID '" + vabf.id.to_s +
                             "'! Hier fehlt der Staat (vbsstaat) des Verwertungs-/Beseitigungsstandortes!")
                  zeile = ['Fehler', 530, "Fehlender Staat des Verwertungs-/Beseitigungsstandortes",
                           b.land_id, b.jahr, Stnd.now, b.kennnr, betriebsname,
                           "Fehlender Staat des Verwertungs-/Beseitigungsstandortes (#{wastetypecode}, #{wastetreatmentcode})",
                           b.id, vabf.id].to_csv(:col_sep => ';', :row_sep => nil)
                  @log_table.error(zeile)
                end
                # raise Exception.new, "ERROR. Feature not implemented yet."
              end
              
            end # end HWOC
          else
            # Geheim =============================================================
            simple_node_with_content_creator(transfer, "rsm:ConfidentialIndicator", "true")
            
            vagk = VAbfallGK.where("state = 0 and p_vabf_id = ?", vabf.id).first
            
            unless vagk == nil
              vv = VertraulichkeitsV.find(vagk.geheimk_id)
              unless vv == nil
                translate_de2eu("ConfidentialCode", vv.schl, transfer,
                                "rsm:ConfidentialCode", b.kennnr, @log)
                simple_node_with_content_creator(transfer,
                                                 "rsm:RemarkText", vabf.geheimd) if vabf.geheimd?
              end
            end
            
          end
        end # Abfall-Schleife
      end
    end # export_abfaelle
    
    
    def export
      @log = Logger.new(@logfile)
      @log_designation = Logger.new(@log_designation_file)
      @log_table = Logger.new(@log_table_file)
      zeile = ['Status', 'Typ', 'Typbeschreibung', 'Land', 'Jahr', 'Zeitpunkt', 'Kennnr',
               'Betriebsname', 'Fehlerbenennung', 'p_betrieb.id', 'sub_id'].to_csv(:col_sep => ';', :row_sep => nil)
      @log_table.error(zeile)

      # AAAACHTUNG! TEMPORÄR auf false für Wunsch der KOM (mlt, 2012-04-24), sonst immer true:
      unterdruecke_zu_unterdrueckende = true;

      if unterdruecke_zu_unterdrueckende == true then
        if @land_id == 0
          # betriebe = Betrieb.find(:all,
          #   :select => "*, st_x(geo_wgs84) as laengengrad, st_y(geo_wgs84) as breitengrad",
          #   :conditions => ["state = 0 and xml3export = 1 and process is null and jahr = '?'", @year],
          #   :order => "land_id, kennnr")
          betriebe = Betrieb.select("*, st_x(geo_wgs84) as laengengrad, st_y(geo_wgs84) as breitengrad").
                     where("state = 0 and xml3export = 1 and process is null and jahr = '?'", @year).
                     order("land_id, kennnr")
        else
          # betriebe = Betrieb.find(:all,
          #   :select => "*, st_x(geo_wgs84) as laengengrad, st_y(geo_wgs84) as breitengrad",
          #   :conditions => ["state = 0 and xml3export = 1 and land_id = ? and process is null and jahr = '?'", @land_id, @year],
          #   :order => "kennnr")
          betriebe = Betrieb.select("*, st_x(geo_wgs84) as laengengrad, st_y(geo_wgs84) as breitengrad").
                     where("state = 0 and xml3export = 1 and land_id = ? and process is null and jahr = '?'", @land_id, @year).
                     order("kennnr")
        end
      else
        # Export MIT den sonst eigentlich zu unterdrückenden Betrieben:
        if @land_id == 0
          # betriebe = Betrieb.find(:all,
          #   :select => "*, st_x(geo_wgs84) as laengengrad, st_y(geo_wgs84) as breitengrad",
          #   :conditions => ["state = 0 and process is null and jahr = '?'", @year],
          #   :order => "land_id, kennnr")
          betriebe = Betrieb.select("*, st_x(geo_wgs84) as laengengrad, st_y(geo_wgs84) as breitengrad").
                     where("state = 0 and process is null and jahr = '?'", @year).
                     order("land_id, kennnr")
        else
          # betriebe = Betrieb.find(:all,
          #   :select => "*, st_x(geo_wgs84) as laengengrad, st_y(geo_wgs84) as breitengrad",
          #   :conditions => ["state = 0 and land_id = ? and process is null and jahr = '?'", @land_id, @year],
          #   :order => "kennnr")
          betriebe = Betrieb.select("*, st_x(geo_wgs84) as laengengrad, st_y(geo_wgs84) as breitengrad").
                     where("state = 0 and land_id = ? and process is null and jahr = '?'", @land_id, @year).
                     order("kennnr")
        end
      end

      
      if betriebe.length > 0
        betriebe.length.times do |nummer|
          b = betriebe[nummer]
          puts "#{nummer}) Kennnr. <#{b.attributes['kennnr']}>"
          betriebsname = b.name1.to_s
          if b.name2
            betriebsname = betriebsname + " " + b.name2
          end

          export_betrieb_einzeln( b, betriebsname )

        end # do betriebe
      end # if betriebe

      @log.close
      @log_designation.close
      @log_table.close
      # @odoc.save(@xmlfile, true)
      @odoc.save(@xmlfile, :indent => true, :encoding => XML::Encoding::UTF_8)

      puts "XML-File " + @xmlfile + " wurde geschrieben!"

#      IO.popen("cd #{@path}; zip -r xml3-#{@year}-revX.zip #{@log} #{@log_designation} #{@log_table} #{@xmlfile}") {
#        |f| $zip_out = f.gets.chomp; puts $zip_out
#      }

    end # export

  end # class Exporter


  class ToGoogleEarth
    include Xml_utilities

    def initialize (xmlfile, jahr, land_id)
      odoc = XML::Document.new('1.0')
      odoc.encoding = 'UTF-8'
      odoc.root = XML::Node.new('kml')

      kml = odoc.root
      kml['xmlns'] = "http://earth.google.com/kml/2.2".to_u

      doc = simple_node_creator(kml, "Document")
      simple_node_with_content_creator(doc, "name", "PRTR-Facilities")
      simple_node_with_content_creator(doc, "open", "1")
      simple_node_with_content_creator(doc, "visibility", "1")

      style = simple_node_with_attribut_creator(doc, "Style", "id", "pushpin-gross")
      iconstyle = simple_node_creator(style, "IconStyle")
      simple_node_with_content_creator(iconstyle, "scale", "1.3")
      icon = simple_node_creator(iconstyle, "Icon")
      simple_node_with_content_creator(icon, "href", "http://maps.google.com/mapfiles/kml/pushpin/ylw-pushpin.png")

      style = simple_node_with_attribut_creator(doc, "Style", "id", "pushpin-klein")
      iconstyle = simple_node_creator(style, "IconStyle")
      simple_node_with_content_creator(iconstyle, "scale", "1.1")
      icon = simple_node_creator(iconstyle, "Icon")
      simple_node_with_content_creator(icon, "href", "http://maps.google.com/mapfiles/kml/pushpin/ylw-pushpin.png")

      stylemap = simple_node_with_attribut_creator(doc, "StyleMap", "id", "pushpin")
      pair = simple_node_creator(stylemap, "Pair")
      simple_node_with_content_creator(pair, "key", "normal")
      simple_node_with_content_creator(pair, "styleUrl", "#pushpin-klein")

      pair = simple_node_creator(stylemap, "Pair")
      simple_node_with_content_creator(pair, "key", "highlight")
      simple_node_with_content_creator(pair, "styleUrl", "#pushpin-gross")

      if land_id == 0
        betriebe = Betrieb.select("*, st_x(geo_wgs84) as laengengrad, st_y(geo_wgs84) as breitengrad").
                   where("state = 0 and process is null and jahr = ?", jahr).
                   order("ido")
      else
        betriebe = Betrieb.select("*, st_x(geo_wgs84) as laengengrad, st_y(geo_wgs84) as breitengrad").
                   where("state = 0 and process is null and jahr = ? and land_id = ?", jahr, land_id).
                   order("ido")
      end

      if betriebe.length > 0
        betriebe.length.times do |b|
          b = betriebe[b]

          if b.laengengrad?
            placemark = simple_node_creator(doc, "Placemark")

            simple_node_with_content_creator(placemark, "name", "%05g" % b.ido )

            d = "<table><tr><td valign=\"top\">Kennnummer:</td><td>" + b.kennnr + "</td></tr>"
            d << "<tr><td valign=\"top\">Name 1:</td><td valign=\"top\">" + b.name1.to_s + "</td></tr>"
            d << "<tr><td valign=\"top\">Name 2:</td><td valign=\"top\">" + b.name2.to_s + "</td></tr></table>"

            taetigkeiten = Taetigkeit.where("state = 0 and p_betrieb_id = ?", b.id).
                           order("htprtr desc")

            if taetigkeiten.length > 0
              d << "<br><table border=\"1\"><tr><th>Tätigkeit</th><th>Ist Haupttätigkeit</th></tr>"
              taetigkeiten.length.times do |t|
                taet = taetigkeiten[t]

                if taet.nrprtr_id?
                  prtr = PRTRTaetigkeit.find(taet.nrprtr_id)

                  unless prtr == nil
                    d << "<tr><td>" + prtr.ktext + "</td>"
                    if taet.htprtr == 1
                      d << "<td>Ja</td></tr>"
                    else
                      d << "<td>Nein</td></tr>"
                    end
                  end
                end
              end
              d << "</table>"
            end

            simple_node_with_content_creator(placemark, "description", d)
            simple_node_with_content_creator(placemark, "open", "1")
            simple_node_with_content_creator(placemark, "visibility", "1")
            simple_node_with_content_creator(placemark, "styleUrl", "#pushpin")

            point = simple_node_creator(placemark, "Point")
            simple_node_with_content_creator(point, "coordinates",
              b.laengengrad.to_s + ", " + b.breitengrad.to_s + ", 0")
          end
        end
      end

      odoc.save(xmlfile, true)
    end
  end


  def export_year(jahr)
    (0..0).each do |land_id|
      puts "Ich verarbeite Nummer #{land_id}"
      exporter = Exporter.new(
        "#{$PROJECT_PATH}/daten/xml/xml3_exports_#{jahr.to_s}",
        "../data/PollutantReleaseAndTransferReport_2p0.xsd",
        jahr.to_i, "Umweltbundesamt", "Wörlitzer Platz", "1", "Dessau-Roßlau", "D-06844",
        "+49 (0)340 2103-2780", "+49 (0)340 2104-2780", "joachim.heidemeier@uba.de",
        "Joachim Heidemeier", land_id)
      if exporter.prepare
        exporter.export
      end
    end
    puts "===== Alle Länder #{jahr.to_s} verarbeitet. ====="
  end

#  $export_hinweise = false

=begin
  export_year(2007)
  export_year(2008)
  export_year(2009)
  export_year(2010)
  export_year(2011)
  export_year(2012)
  export_year(2013)
=end
  export_year(2014)


  #ge = ToGoogleEarth.new("C:/Temp/prtr.kml", 2008, 0)
  #break

rescue Exception => rumms
  puts "Unexpected Exc <#{rumms}>"
  print rumms.backtrace.join("\n")
end
